SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

IF EXISTS (SELECT * FROM dbo.sysobjects WHERE id = object_id(N'[dbo].[cs_Wiki_GetPageComments]') AND OBJECTPROPERTY(id, N'IsProcedure') = 1)
DROP PROCEDURE [dbo].[cs_Wiki_GetPageComments]
GO

CREATE PROCEDURE [dbo].[cs_Wiki_GetPageComments]
(
    @pageId int,
    @pageIndex int,
    @pageSize int,
    @sortBy nvarchar (256), 
    @sortOrder nvarchar (256), 
    @totalCount int OUTPUT
)
AS
BEGIN

	DECLARE @rowsToReturn int
	DECLARE @pageLowerBound int
	DECLARE @pageUpperBound int

	/* how many comments are there in total? doing this first to offset other processing */
	SET @totalCount = (
		SELECT
			count(*) as CountOfPageComments
		FROM
			cs_Wiki_PageComments PC
		WHERE
			PC.PageId = @pageId
		)

	/* we want nothing but the total count */
	if @pageSize = 0
	begin
		SELECT
			PC.PageCommentId,
			PC.PageId,
			PC.UserId,
			PC.Body,
			PC.FormattedBody,
			PC.RatingCount,
			PC.RatingSum,
			PC.PublishedUtcDate
		FROM
			cs_Wiki_PageComments PC
		WHERE
			1 = 0

		/* exit prematurely */
		return 0
	end

	/* we actually want data */

	exec [dbo].[cs_system_CalculatePagingInformation]
		@pageIndex, @pageSize
		, @rowsToReturn output, @pageLowerBound output, @pageUpperBound output

	declare @Results table
	(
		IndexId int IDENTITY(1, 1) NOT NULL,
		PageCommentId int NOT NULL
	)

	IF ((@sortBy is NULL) OR (LEN(@sortBy) = 0))
		BEGIN
			SET @sortBy = N'PublishedDate';
		END

	IF ((@sortOrder is NULL) OR (LEN(@sortOrder) = 0))
		BEGIN
			SET @sortOrder = N'Ascending';
		END

	/* setting the minimum number of records that needs to come into this table */
	SET ROWCOUNT @rowsToReturn

	IF (@sortBy = N'RatingAverage') 
		BEGIN
			-- By RatingAverage (desc only, no asc)
			INSERT INTO @Results
				(PageCommentId)
			SELECT
				PC.PageCommentId
			FROM
				cs_Wiki_PageComments PC
			WHERE
				PC.PageId = @pageId
			ORDER BY
				(RatingSum / (RatingCount + .1)) DESC, RatingCount, RatingSum
		END
	ELSE IF (@sortBy=N'MostRecent' OR (@sortBy = N'PublishedDate' AND @sortOrder = N'Descending'))
		BEGIN
			-- By PublishedDate Desc
			INSERT INTO @Results
				(PageCommentId)
			SELECT
				PC.PageCommentId
			FROM
				cs_Wiki_PageComments PC
			WHERE
				PC.PageId = @pageId
			ORDER BY
				PublishedUtcDate DESC
		END
	ELSE 
		BEGIN
			-- The Default by PublishedDate asc
			INSERT INTO @Results
				(PageCommentId)
			SELECT
				PC.PageCommentId
			FROM
				cs_Wiki_PageComments PC
			WHERE
				PC.PageId = @pageId
			ORDER BY
				PublishedUtcDate ASC
		END

	SET ROWCOUNT @pageSize

	SELECT
		PC.PageCommentId,
		PC.PageId,
		PC.UserId,
		PC.Body,
		PC.FormattedBody,
		PC.RatingCount,
		PC.RatingSum,
		PC.PublishedUtcDate
	FROM
		cs_Wiki_PageComments PC
		INNER JOIN @Results R ON PC.PageCommentId = R.PageCommentId
	WHERE
		R.IndexId > @pageLowerBound
		AND R.IndexId < @pageUpperBound

END

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

GRANT EXECUTE ON [dbo].[cs_Wiki_GetPageComments] TO PUBLIC
GO
